<?php
// Start session to display errors or success messages
session_start();

// Database connection parameters (adjust these as per your environment)
$servername = "localhost";
$username = "root";
$password = "";
$dbname = "sports_warehouse";

// Create a connection to the database
$conn = new mysqli($servername, $username, $password, $dbname);

// Check connection
if ($conn->connect_error) {
    die("Connection failed: " . $conn->connect_error);
}

// Function to sanitize user input
function sanitize_input($data) {
    return htmlspecialchars(stripslashes(trim($data)));
}

// Initialize variables for storing user input and error messages
$name = $email = $password = $confirm_password = "";
$role = 'customer'; // Default role
$errors = [];

if ($_SERVER["REQUEST_METHOD"] == "POST") {

    // Sanitize and validate full name
    if (empty($_POST["name"])) {
        $errors[] = "Full name is required.";
    } else {
        $name = sanitize_input($_POST["name"]);
    }

    // Sanitize and validate email
    if (empty($_POST["email"])) {
        $errors[] = "Email is required.";
    } else {
        $email = filter_var(sanitize_input($_POST["email"]), FILTER_VALIDATE_EMAIL);
        if (!$email) {
            $errors[] = "Invalid email format.";
        }
    }

    // Sanitize and validate password
    if (empty($_POST["password"])) {
        $errors[] = "Password is required.";
    } else {
        $password = sanitize_input($_POST["password"]);
        if (strlen($password) < 8) {
            $errors[] = "Password must be at least 8 characters long.";
        }
    }

    // Confirm password
    if (empty($_POST["confirm-password"])) {
        $errors[] = "Confirm password is required.";
    } else {
        $confirm_password = sanitize_input($_POST["confirm-password"]);
        if ($password !== $confirm_password) {
            $errors[] = "Passwords do not match.";
        }
    }

    // Sanitize and validate role
    if (isset($_POST["role"])) {
        $role = sanitize_input($_POST["role"]);
        if (!in_array($role, ['admin', 'customer'])) {
            $errors[] = "Invalid role selected.";
        }
    }

    // Check if email already exists in the database
    if (empty($errors)) {
        $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $stmt->store_result();

        if ($stmt->num_rows > 0) {
            $errors[] = "Email already exists.";
        }

        $stmt->close();
    }

    // If no errors, proceed with registration
    if (empty($errors)) {
        // Hash the password securely
        $hashed_password = password_hash($password, PASSWORD_BCRYPT);

        // Insert user details into the database
        $stmt = $conn->prepare("INSERT INTO users (name, email, password, role) VALUES (?, ?, ?, ?)");
        $stmt->bind_param("ssss", $name, $email, $hashed_password, $role);

        if ($stmt->execute()) {
            $_SESSION['success'] = "Registration successful. You can now login.";
            header("Location: login.php");
            exit();
        } else {
            $errors[] = "Error: Could not register user.";
        }

        $stmt->close();
    }

    // Close the database connection
    $conn->close();
}

// If there are errors, store them in the session to display on the registration page
if (!empty($errors)) {
    $_SESSION['errors'] = $errors;
    header("Location: register.php");
    exit();
}
?>
